<?php
/**
 * JWT (JSON Web Token) Class
 * Token generation and validation for authentication
 */

class JWT {
    /**
     * Generate JWT token
     */
    public static function encode(array $payload, int $expiry = null): string {
        $expiry = $expiry ?? JWT_EXPIRY;
        
        // Header
        $header = [
            'typ' => 'JWT',
            'alg' => JWT_ALGORITHM
        ];
        
        // Payload
        $payload['iat'] = time();
        $payload['exp'] = time() + $expiry;
        $payload['iss'] = JWT_ISSUER;
        
        // Encode
        $headerEncoded = self::base64UrlEncode(json_encode($header));
        $payloadEncoded = self::base64UrlEncode(json_encode($payload));
        
        // Signature
        $signature = hash_hmac('sha256', "$headerEncoded.$payloadEncoded", JWT_SECRET, true);
        $signatureEncoded = self::base64UrlEncode($signature);
        
        return "$headerEncoded.$payloadEncoded.$signatureEncoded";
    }
    
    /**
     * Decode and validate JWT token
     */
    public static function decode(string $token): ?array {
        $parts = explode('.', $token);
        
        if (count($parts) !== 3) {
            return null;
        }
        
        list($headerEncoded, $payloadEncoded, $signatureEncoded) = $parts;
        
        // Verify signature
        $signature = self::base64UrlDecode($signatureEncoded);
        $expectedSignature = hash_hmac('sha256', "$headerEncoded.$payloadEncoded", JWT_SECRET, true);
        
        if (!hash_equals($expectedSignature, $signature)) {
            Logger::warning('JWT signature verification failed');
            return null;
        }
        
        // Decode payload
        $payload = json_decode(self::base64UrlDecode($payloadEncoded), true);
        
        if (!$payload) {
            return null;
        }
        
        // Check expiration
        if (isset($payload['exp']) && $payload['exp'] < time()) {
            return null;
        }
        
        // Check issuer
        if (isset($payload['iss']) && $payload['iss'] !== JWT_ISSUER) {
            return null;
        }
        
        return $payload;
    }
    
    /**
     * Generate refresh token
     */
    public static function generateRefreshToken(): string {
        return bin2hex(random_bytes(32));
    }
    
    /**
     * Hash refresh token for storage
     */
    public static function hashRefreshToken(string $token): string {
        return hash('sha256', $token);
    }
    
    /**
     * Base64 URL encode
     */
    private static function base64UrlEncode(string $data): string {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
    
    /**
     * Base64 URL decode
     */
    private static function base64UrlDecode(string $data): string {
        return base64_decode(strtr($data, '-_', '+/'));
    }
    
    /**
     * Extract token from Authorization header
     */
    public static function fromRequest(Request $request): ?string {
        return $request->bearerToken();
    }
    
    /**
     * Verify and decode token from request
     */
    public static function verifyRequest(Request $request): ?array {
        $token = self::fromRequest($request);
        
        if (!$token) {
            return null;
        }
        
        return self::decode($token);
    }
}
