<?php
/**
 * Xendit Payment Gateway Service
 * Implements Xendit API integration (NO WebView)
 */
class XenditService
{
    private $apiKey;
    private $baseUrl;
    private $logger;

    public function __construct()
    {
        $this->apiKey = XENDIT_API_KEY;
        $this->baseUrl = 'https://api.xendit.co';
        $this->logger = Logger::getInstance();
    }

    /**
     * Create invoice
     */
    public function createInvoice($data)
    {
        try {
            $payload = [
                'external_id' => $data['external_id'],
                'amount' => $data['amount'],
                'description' => $data['description'] ?? 'Payment for invoice',
                'invoice_duration' => $data['duration'] ?? 86400, // 24 hours default
                'customer' => [
                    'given_names' => $data['customer_name'],
                    'email' => $data['customer_email'],
                    'mobile_number' => $data['customer_phone'] ?? null,
                ],
                'customer_notification_preference' => [
                    'invoice_created' => ['email', 'sms'],
                    'invoice_reminder' => ['email', 'sms'],
                    'invoice_paid' => ['email', 'sms'],
                ],
                'success_redirect_url' => $data['success_url'] ?? null,
                'failure_redirect_url' => $data['failure_url'] ?? null,
                'currency' => 'IDR',
                'items' => $data['items'] ?? [],
            ];

            $response = $this->makeRequest('POST', '/v2/invoices', $payload);

            $this->logger->info('Xendit invoice created', [
                'external_id' => $data['external_id'],
                'invoice_id' => $response['id']
            ]);

            return [
                'success' => true,
                'invoice_id' => $response['id'],
                'invoice_url' => $response['invoice_url'],
                'external_id' => $response['external_id'],
                'amount' => $response['amount'],
                'expiry_date' => $response['expiry_date'],
                'status' => $response['status'],
            ];

        } catch (Exception $e) {
            $this->logger->error('Xendit invoice creation failed', [
                'external_id' => $data['external_id'],
                'error' => $e->getMessage()
            ]);
            throw new PaymentException('Failed to create Xendit invoice: ' . $e->getMessage());
        }
    }

    /**
     * Get invoice details
     */
    public function getInvoice($invoiceId)
    {
        try {
            $response = $this->makeRequest('GET', "/v2/invoices/{$invoiceId}");

            return [
                'success' => true,
                'invoice_id' => $response['id'],
                'external_id' => $response['external_id'],
                'status' => $response['status'],
                'amount' => $response['amount'],
                'paid_amount' => $response['paid_amount'] ?? 0,
                'paid_at' => $response['paid_at'] ?? null,
                'expiry_date' => $response['expiry_date'],
            ];

        } catch (Exception $e) {
            $this->logger->error('Failed to get Xendit invoice', [
                'invoice_id' => $invoiceId,
                'error' => $e->getMessage()
            ]);
            throw new PaymentException('Failed to get invoice details');
        }
    }

    /**
     * Create disbursement (for refunds or payouts)
     */
    public function createDisbursement($data)
    {
        try {
            $payload = [
                'external_id' => $data['external_id'],
                'bank_code' => $data['bank_code'],
                'account_holder_name' => $data['account_holder_name'],
                'account_number' => $data['account_number'],
                'description' => $data['description'] ?? 'Disbursement',
                'amount' => $data['amount'],
            ];

            $response = $this->makeRequest('POST', '/disbursements', $payload);

            $this->logger->info('Xendit disbursement created', [
                'external_id' => $data['external_id'],
                'disbursement_id' => $response['id']
            ]);

            return [
                'success' => true,
                'disbursement_id' => $response['id'],
                'external_id' => $response['external_id'],
                'status' => $response['status'],
                'amount' => $response['amount'],
            ];

        } catch (Exception $e) {
            $this->logger->error('Xendit disbursement failed', [
                'external_id' => $data['external_id'],
                'error' => $e->getMessage()
            ]);
            throw new PaymentException('Failed to create disbursement');
        }
    }

    /**
     * Get available payment channels
     */
    public function getPaymentChannels()
    {
        return [
            [
                'code' => 'CREDIT_CARD',
                'name' => 'Credit Card',
                'type' => 'card',
                'admin_fee' => 0,
                'admin_fee_type' => 'percentage',
                'admin_fee_value' => 2.9,
            ],
            [
                'code' => 'BCA',
                'name' => 'Bank BCA',
                'type' => 'virtual_account',
                'admin_fee' => 4000,
                'admin_fee_type' => 'flat',
            ],
            [
                'code' => 'BNI',
                'name' => 'Bank BNI',
                'type' => 'virtual_account',
                'admin_fee' => 4000,
                'admin_fee_type' => 'flat',
            ],
            [
                'code' => 'BRI',
                'name' => 'Bank BRI',
                'type' => 'virtual_account',
                'admin_fee' => 4000,
                'admin_fee_type' => 'flat',
            ],
            [
                'code' => 'MANDIRI',
                'name' => 'Bank Mandiri',
                'type' => 'virtual_account',
                'admin_fee' => 4000,
                'admin_fee_type' => 'flat',
            ],
            [
                'code' => 'ALFAMART',
                'name' => 'Alfamart',
                'type' => 'retail',
                'admin_fee' => 2500,
                'admin_fee_type' => 'flat',
            ],
            [
                'code' => 'INDOMARET',
                'name' => 'Indomaret',
                'type' => 'retail',
                'admin_fee' => 2500,
                'admin_fee_type' => 'flat',
            ],
            [
                'code' => 'OVO',
                'name' => 'OVO',
                'type' => 'ewallet',
                'admin_fee' => 0,
                'admin_fee_type' => 'percentage',
                'admin_fee_value' => 2,
            ],
            [
                'code' => 'DANA',
                'name' => 'DANA',
                'type' => 'ewallet',
                'admin_fee' => 0,
                'admin_fee_type' => 'percentage',
                'admin_fee_value' => 2,
            ],
            [
                'code' => 'LINKAJA',
                'name' => 'LinkAja',
                'type' => 'ewallet',
                'admin_fee' => 0,
                'admin_fee_type' => 'percentage',
                'admin_fee_value' => 2,
            ],
        ];
    }

    /**
     * Verify webhook signature
     */
    public function verifyWebhookSignature($token, $payload)
    {
        $expectedToken = hash_hmac('sha256', json_encode($payload), XENDIT_WEBHOOK_TOKEN);
        return hash_equals($expectedToken, $token);
    }

    /**
     * Make HTTP request to Xendit API
     */
    private function makeRequest($method, $endpoint, $data = null)
    {
        $url = $this->baseUrl . $endpoint;
        
        $ch = curl_init();
        
        $headers = [
            'Authorization: Basic ' . base64_encode($this->apiKey . ':'),
            'Content-Type: application/json',
        ];
        
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        } elseif ($method === 'PUT') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        } elseif ($method === 'DELETE') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        if ($error) {
            throw new Exception("cURL Error: {$error}");
        }
        
        $responseData = json_decode($response, true);
        
        if ($httpCode >= 400) {
            $errorMessage = $responseData['message'] ?? $responseData['error_code'] ?? 'Unknown error';
            throw new Exception("Xendit API Error ({$httpCode}): {$errorMessage}");
        }
        
        return $responseData;
    }
}
