<?php
/**
 * Payment Gateway Service
 * Abstraction layer for payment gateways
 */
class PaymentGatewayService
{
    private $paymentRepo;
    private $invoiceRepo;
    private $xenditService;
    private $tripayService;
    private $logger;

    public function __construct()
    {
        $this->paymentRepo = new PaymentRepository();
        $this->invoiceRepo = new InvoiceRepository();
        $this->xenditService = new XenditService();
        $this->tripayService = new TripayService();
        $this->logger = Logger::getInstance();
    }

    /**
     * Create payment
     */
    public function createPayment($invoiceId, $gatewayId, $channelId = null)
    {
        try {
            // Get invoice
            $invoice = $this->invoiceRepo->findById($invoiceId);
            
            if (!$invoice) {
                throw new NotFoundException('Invoice not found');
            }

            if ($invoice->isPaid()) {
                throw new PaymentException('Invoice is already paid');
            }

            // Get gateway
            $gateway = $this->getGateway($gatewayId);
            
            if (!$gateway) {
                throw new NotFoundException('Payment gateway not found');
            }

            // Create payment record
            $paymentData = [
                'invoice_id' => $invoiceId,
                'gateway_id' => $gatewayId,
                'channel_id' => $channelId,
                'amount' => $invoice->getTotalAmount(),
                'admin_fee' => 0, // Will be updated from gateway response
                'total_amount' => $invoice->getTotalAmount(),
                'status' => 'pending',
                'expired_at' => date('Y-m-d H:i:s', strtotime('+24 hours'))
            ];

            $payment = $this->paymentRepo->create($paymentData);

            // Process with appropriate gateway
            $gatewayResponse = null;
            
            if ($gateway['code'] === 'XENDIT') {
                $gatewayResponse = $this->processXenditPayment($payment, $invoice);
            } elseif ($gateway['code'] === 'TRIPAY') {
                $gatewayResponse = $this->processTripayPayment($payment, $invoice, $channelId);
            }

            // Update payment with gateway response
            if ($gatewayResponse && $gatewayResponse['success']) {
                $updateData = [
                    'payment_url' => $gatewayResponse['invoice_url'] ?? $gatewayResponse['checkout_url'] ?? null,
                    'external_id' => $gatewayResponse['invoice_id'] ?? $gatewayResponse['transaction_id'] ?? null,
                ];
                
                if (isset($gatewayResponse['qr_string'])) {
                    $updateData['qr_string'] = $gatewayResponse['qr_string'];
                }
                if (isset($gatewayResponse['va_number']) || isset($gatewayResponse['pay_code'])) {
                    $updateData['va_number'] = $gatewayResponse['va_number'] ?? $gatewayResponse['pay_code'];
                }
                if (isset($gatewayResponse['fee'])) {
                    $updateData['admin_fee'] = $gatewayResponse['fee'];
                    $updateData['total_amount'] = $payment->getAmount() + $gatewayResponse['fee'];
                }
                
                $this->paymentRepo->update($payment->getId(), $updateData);
            }

            $this->logger->info('Payment created', [
                'payment_id' => $payment->getId(),
                'invoice_id' => $invoiceId,
                'gateway' => $gateway['code']
            ]);

            return $this->paymentRepo->findById($payment->getId());

        } catch (NotFoundException | PaymentException $e) {
            throw $e;
        } catch (Exception $e) {
            $this->logger->error('Payment creation failed', [
                'invoice_id' => $invoiceId,
                'error' => $e->getMessage()
            ]);
            throw new PaymentException('Failed to create payment');
        }
    }

    /**
     * Process Xendit payment
     */
    private function processXenditPayment($payment, $invoice)
    {
        $customer = $invoice->getCustomer();
        
        $data = [
            'external_id' => $payment->getPaymentCode(),
            'amount' => $payment->getAmount(),
            'description' => "Payment for Invoice {$invoice->getInvoiceNumber()}",
            'customer_name' => $customer['full_name'] ?? 'Customer',
            'customer_email' => $customer['email'] ?? '',
            'customer_phone' => $customer['phone'] ?? null,
            'duration' => 86400, // 24 hours
            'success_url' => APP_URL . '/payment/success',
            'failure_url' => APP_URL . '/payment/failure',
            'items' => []
        ];

        return $this->xenditService->createInvoice($data);
    }

    /**
     * Process Tripay payment
     */
    private function processTripayPayment($payment, $invoice, $channelId)
    {
        $customer = $invoice->getCustomer();
        
        // Get channel code
        $channelCode = null;
        if ($channelId) {
            $db = Database::getInstance();
            $channelQuery = "SELECT code FROM payment_channels WHERE id = ?";
            $channelResult = $db->query($channelQuery, [$channelId]);
            $channelCode = $channelResult[0]['code'] ?? null;
        }
        
        if (!$channelCode) {
            return ['success' => false, 'message' => 'Payment channel is required for Tripay'];
        }
        
        $data = [
            'external_id' => $payment->getPaymentCode(),
            'payment_method' => $channelCode,
            'amount' => $payment->getAmount(),
            'description' => "Payment for Invoice {$invoice->getInvoiceNumber()}",
            'customer_name' => $customer['full_name'] ?? 'Customer',
            'customer_email' => $customer['email'] ?? '',
            'customer_phone' => $customer['phone'] ?? null,
            'duration' => 86400, // 24 hours
            'success_url' => APP_URL . '/payment/success'
        ];

        return $this->tripayService->createTransaction($data);
    }

    /**
     * Get payment by ID
     */
    public function getPayment($paymentId)
    {
        $payment = $this->paymentRepo->findById($paymentId);
        
        if (!$payment) {
            throw new NotFoundException('Payment not found');
        }

        return $payment;
    }

    /**
     * Process webhook
     */
    public function processWebhook($gatewayId, $payload, $signature = null)
    {
        try {
            $gateway = $this->getGateway($gatewayId);
            
            if (!$gateway) {
                throw new NotFoundException('Payment gateway not found');
            }

            // Verify signature
            if ($gateway['code'] === 'XENDIT') {
                if (!$this->xenditService->verifyWebhookSignature($signature, $payload)) {
                    throw new PaymentException('Invalid webhook signature');
                }
                
                return $this->processXenditWebhook($payload);
            } elseif ($gateway['code'] === 'TRIPAY') {
                if (!$this->tripayService->verifyWebhookSignature($signature, $payload)) {
                    throw new PaymentException('Invalid webhook signature');
                }
                
                return $this->processTripayWebhook($payload);
            }

            throw new PaymentException('Unsupported gateway');

        } catch (Exception $e) {
            $this->logger->error('Webhook processing failed', [
                'gateway_id' => $gatewayId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Process Xendit webhook
     */
    private function processXenditWebhook($payload)
    {
        // Find payment by external ID
        $payment = $this->paymentRepo->findByExternalId($payload['id']);
        
        if (!$payment) {
            $this->logger->warning('Payment not found for webhook', ['external_id' => $payload['id']]);
            return false;
        }

        // Update payment status based on webhook
        $status = strtolower($payload['status']);
        
        if ($status === 'paid' || $status === 'settled') {
            $this->paymentRepo->markAsPaid($payment->getId());
            
            // Mark invoice as paid
            $billingService = new BillingService();
            $billingService->markAsPaid($payment->getInvoiceId());
            
            $this->logger->info('Payment marked as paid via webhook', [
                'payment_id' => $payment->getId()
            ]);
        } elseif ($status === 'expired') {
            $this->paymentRepo->update($payment->getId(), ['status' => 'expired']);
        }

        // Log webhook
        $this->paymentRepo->recordWebhook(
            $payment->getId(),
            $payment->getGatewayId(),
            $payload,
            null
        );

        return true;
    }

    /**
     * Process Tripay webhook
     */
    private function processTripayWebhook($payload)
    {
        // Find payment by external ID (merchant_ref)
        $merchantRef = $payload['merchant_ref'] ?? null;
        
        if (!$merchantRef) {
            $this->logger->warning('Merchant ref not found in Tripay webhook');
            return false;
        }
        
        $payment = $this->paymentRepo->findByPaymentCode($merchantRef);
        
        if (!$payment) {
            $this->logger->warning('Payment not found for Tripay webhook', ['merchant_ref' => $merchantRef]);
            return false;
        }

        // Update payment status based on webhook
        $status = strtolower($payload['status']);
        
        if ($status === 'paid') {
            $this->paymentRepo->markAsPaid($payment->getId());
            
            // Mark invoice as paid
            $billingService = new BillingService();
            $billingService->markAsPaid($payment->getInvoiceId());
            
            // Send notification
            $notificationService = new NotificationService();
            $notificationService->sendPaymentConfirmation(
                $payment->getCustomerId(),
                $payment->getId(),
                $payment->getAmount()
            );
            
            $this->logger->info('Payment marked as paid via Tripay webhook', [
                'payment_id' => $payment->getId()
            ]);
        } elseif ($status === 'expired' || $status === 'failed') {
            $this->paymentRepo->update($payment->getId(), ['status' => $status]);
        }

        // Log webhook
        $this->paymentRepo->recordWebhook(
            $payment->getId(),
            $payment->getGatewayId(),
            $payload,
            null
        );

        return true;
    }

    /**
     * Get available payment channels
     */
    public function getPaymentChannels($gatewayId = null)
    {
        if ($gatewayId) {
            $gateway = $this->getGateway($gatewayId);
            
            if ($gateway && $gateway['code'] === 'XENDIT') {
                return $this->xenditService->getPaymentChannels();
            } elseif ($gateway && $gateway['code'] === 'TRIPAY') {
                return $this->tripayService->getPaymentChannels();
            }
        }

        // Return all channels from all gateways
        $channels = [];
        $channels = array_merge($channels, $this->xenditService->getPaymentChannels());
        $channels = array_merge($channels, $this->tripayService->getPaymentChannels());
        return $channels;
    }

    /**
     * Get gateway by ID
     */
    private function getGateway($gatewayId)
    {
        $db = Database::getInstance();
        $query = "SELECT * FROM payment_gateways WHERE id = ? AND is_active = 1";
        $result = $db->query($query, [$gatewayId]);
        
        return $result[0] ?? null;
    }

    /**
     * Get all active gateways
     */
    public function getActiveGateways()
    {
        $db = Database::getInstance();
        $query = "SELECT * FROM payment_gateways WHERE is_active = 1 ORDER BY name";
        return $db->query($query);
    }
}
