<?php
/**
 * Customer Service
 * Handles customer business logic
 */
class CustomerService
{
    private $customerRepo;
    private $userRepo;
    private $logger;

    public function __construct()
    {
        $this->customerRepo = new CustomerRepository();
        $this->userRepo = new UserRepository();
        $this->logger = Logger::getInstance();
    }

    /**
     * Register new customer with user account
     */
    public function register($data)
    {
        try {
            Database::getInstance()->beginTransaction();

            // Create user account
            $userData = [
                'email' => $data['email'],
                'password' => password_hash($data['password'], PASSWORD_BCRYPT, ['cost' => 10]),
                'full_name' => $data['full_name'],
                'phone' => $data['phone'] ?? null,
                'role_id' => $this->getCustomerRoleId(),
                'status' => 'active'
            ];

            $user = $this->userRepo->create($userData);

            // Create customer profile
            $customerData = [
                'user_id' => $user->getId(),
                'full_name' => $data['full_name'],
                'phone' => $data['phone'] ?? null,
                'address' => $data['address'] ?? null,
                'city' => $data['city'] ?? null,
                'province' => $data['province'] ?? null,
                'postal_code' => $data['postal_code'] ?? null,
                'latitude' => $data['latitude'] ?? null,
                'longitude' => $data['longitude'] ?? null,
                'status' => 'active'
            ];

            $customer = $this->customerRepo->create($customerData);

            Database::getInstance()->commit();

            $this->logger->info('Customer registered successfully', [
                'customer_id' => $customer->getId(),
                'user_id' => $user->getId()
            ]);

            return [
                'user' => $user->toArray(),
                'customer' => $customer->toArray()
            ];

        } catch (Exception $e) {
            Database::getInstance()->rollback();
            $this->logger->error('Customer registration failed', ['error' => $e->getMessage()]);
            throw new DatabaseException('Failed to register customer');
        }
    }

    /**
     * Get customer by user ID
     */
    public function getByUserId($userId)
    {
        $customer = $this->customerRepo->findByUserId($userId);
        
        if (!$customer) {
            throw new NotFoundException('Customer not found');
        }

        return $customer;
    }

    /**
     * Update customer profile
     */
    public function updateProfile($customerId, $data)
    {
        try {
            $customer = $this->customerRepo->findById($customerId);
            
            if (!$customer) {
                throw new NotFoundException('Customer not found');
            }

            $updated = $this->customerRepo->update($customerId, $data);

            $this->logger->info('Customer profile updated', ['customer_id' => $customerId]);

            return $updated;

        } catch (NotFoundException $e) {
            throw $e;
        } catch (Exception $e) {
            $this->logger->error('Customer profile update failed', [
                'customer_id' => $customerId,
                'error' => $e->getMessage()
            ]);
            throw new DatabaseException('Failed to update profile');
        }
    }

    /**
     * Get customers list with pagination
     */
    public function getAll($page = 1, $perPage = 20, $filters = [])
    {
        $customers = $this->customerRepo->getAll($page, $perPage, $filters);
        $total = $this->customerRepo->count($filters);

        return [
            'data' => array_map(function($customer) {
                return $customer->toArray();
            }, $customers),
            'pagination' => [
                'current_page' => $page,
                'per_page' => $perPage,
                'total' => $total,
                'total_pages' => ceil($total / $perPage)
            ]
        ];
    }

    /**
     * Get customers by location (for collector)
     */
    public function getByLocation($latitude, $longitude, $radiusKm = 10)
    {
        $customers = $this->customerRepo->getByLocation($latitude, $longitude, $radiusKm);

        return array_map(function($customer) {
            return $customer->toArray();
        }, $customers);
    }

    /**
     * Get customer role ID
     */
    private function getCustomerRoleId()
    {
        $db = Database::getInstance();
        $query = "SELECT id FROM roles WHERE name = 'customer' LIMIT 1";
        $result = $db->query($query);
        
        return $result[0]['id'] ?? null;
    }
}
