<?php
/**
 * Authentication Service
 * Handles authentication business logic
 */
class AuthService
{
    private $userRepo;
    private $logger;

    public function __construct()
    {
        $this->userRepo = new UserRepository();
        $this->logger = Logger::getInstance();
    }

    /**
     * Login user
     */
    public function login($email, $password, $ipAddress, $userAgent)
    {
        try {
            // Find user by email
            $user = $this->userRepo->findByEmail($email);
            
            if (!$user) {
                $this->logger->warning('Login attempt with invalid email', ['email' => $email]);
                throw new AuthenticationException('Invalid credentials');
            }

            // Check if account is active
            if (!$user->isActive()) {
                throw new AuthenticationException('Account is inactive');
            }

            // Check failed login attempts
            $failedAttempts = $this->userRepo->getFailedLoginAttempts($user->getId());
            
            if ($failedAttempts >= MAX_LOGIN_ATTEMPTS) {
                $this->logger->warning('Account locked due to failed attempts', [
                    'user_id' => $user->getId(),
                    'email' => $email
                ]);
                throw new AuthenticationException('Account locked. Please try again later.');
            }

            // Verify password
            if (!password_verify($password, $user->getPassword())) {
                // Record failed attempt
                $this->userRepo->recordLoginAttempt($user->getId(), $ipAddress, $userAgent, false);
                
                $this->logger->warning('Login attempt with invalid password', [
                    'user_id' => $user->getId(),
                    'email' => $email
                ]);
                
                throw new AuthenticationException('Invalid credentials');
            }

            // Load user with permissions
            $user = $this->userRepo->findByIdWithPermissions($user->getId());

            // Clear failed attempts
            $this->userRepo->clearLoginAttempts($user->getId());

            // Record successful attempt
            $this->userRepo->recordLoginAttempt($user->getId(), $ipAddress, $userAgent, true);

            // Generate tokens
            $accessToken = JWT::encode([
                'user_id' => $user->getId(),
                'email' => $user->getEmail(),
                'role_id' => $user->getRoleId(),
                'exp' => time() + JWT_EXPIRY
            ]);

            $refreshToken = JWT::encode([
                'user_id' => $user->getId(),
                'type' => 'refresh',
                'exp' => time() + JWT_REFRESH_EXPIRY
            ]);

            // Store refresh token
            $this->storeRefreshToken($user->getId(), $refreshToken, $ipAddress, $userAgent);

            $this->logger->info('User logged in successfully', [
                'user_id' => $user->getId(),
                'email' => $email
            ]);

            return [
                'user' => $user->toArray(),
                'access_token' => $accessToken,
                'refresh_token' => $refreshToken,
                'token_type' => 'Bearer',
                'expires_in' => JWT_EXPIRY
            ];

        } catch (AuthenticationException $e) {
            throw $e;
        } catch (Exception $e) {
            $this->logger->error('Login error', [
                'email' => $email,
                'error' => $e->getMessage()
            ]);
            throw new AuthenticationException('Authentication failed');
        }
    }

    /**
     * Register new user
     */
    public function register($data)
    {
        try {
            // Validate email uniqueness
            if ($this->userRepo->findByEmail($data['email'])) {
                throw new ValidationException('Email already exists', ['email' => 'Email is already registered']);
            }

            // Hash password
            $data['password'] = password_hash($data['password'], PASSWORD_BCRYPT, ['cost' => 10]);

            // Set default role (customer)
            if (!isset($data['role_id'])) {
                $data['role_id'] = $this->getCustomerRoleId();
            }

            // Create user
            $user = $this->userRepo->create($data);

            $this->logger->info('New user registered', [
                'user_id' => $user->getId(),
                'email' => $user->getEmail()
            ]);

            return $user;

        } catch (ValidationException $e) {
            throw $e;
        } catch (Exception $e) {
            $this->logger->error('Registration error', [
                'email' => $data['email'] ?? 'unknown',
                'error' => $e->getMessage()
            ]);
            throw new DatabaseException('Registration failed');
        }
    }

    /**
     * Refresh access token
     */
    public function refreshToken($refreshToken)
    {
        try {
            // Decode refresh token
            $payload = JWT::decode($refreshToken);

            if (!isset($payload['type']) || $payload['type'] !== 'refresh') {
                throw new AuthenticationException('Invalid refresh token');
            }

            // Verify refresh token exists in database
            if (!$this->verifyRefreshToken($payload['user_id'], $refreshToken)) {
                throw new AuthenticationException('Refresh token not found or expired');
            }

            // Get user with permissions
            $user = $this->userRepo->findByIdWithPermissions($payload['user_id']);

            if (!$user || !$user->isActive()) {
                throw new AuthenticationException('User not found or inactive');
            }

            // Generate new access token
            $accessToken = JWT::encode([
                'user_id' => $user->getId(),
                'email' => $user->getEmail(),
                'role_id' => $user->getRoleId(),
                'exp' => time() + JWT_EXPIRY
            ]);

            // Generate new refresh token (rotation)
            $newRefreshToken = JWT::encode([
                'user_id' => $user->getId(),
                'type' => 'refresh',
                'exp' => time() + JWT_REFRESH_EXPIRY
            ]);

            // Revoke old refresh token and store new one
            $this->revokeRefreshToken($refreshToken);
            $this->storeRefreshToken($user->getId(), $newRefreshToken, $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT']);

            return [
                'access_token' => $accessToken,
                'refresh_token' => $newRefreshToken,
                'token_type' => 'Bearer',
                'expires_in' => JWT_EXPIRY
            ];

        } catch (AuthenticationException $e) {
            throw $e;
        } catch (Exception $e) {
            $this->logger->error('Token refresh error', ['error' => $e->getMessage()]);
            throw new AuthenticationException('Token refresh failed');
        }
    }

    /**
     * Logout user
     */
    public function logout($userId, $refreshToken = null)
    {
        try {
            if ($refreshToken) {
                $this->revokeRefreshToken($refreshToken);
            }

            $this->logger->info('User logged out', ['user_id' => $userId]);

            return true;

        } catch (Exception $e) {
            $this->logger->error('Logout error', [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Store refresh token in database
     */
    private function storeRefreshToken($userId, $token, $ipAddress, $userAgent)
    {
        $db = Database::getInstance();
        $id = $db->generateUUID();
        $expiresAt = date('Y-m-d H:i:s', time() + JWT_REFRESH_EXPIRY);
        
        $query = "INSERT INTO refresh_tokens 
                  (id, user_id, token, ip_address, user_agent, expires_at, created_at)
                  VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $db->execute($query, [
            $id,
            $userId,
            $token,
            $ipAddress,
            $userAgent,
            $expiresAt,
            date('Y-m-d H:i:s')
        ]);
    }

    /**
     * Verify refresh token exists and is valid
     */
    private function verifyRefreshToken($userId, $token)
    {
        $db = Database::getInstance();
        $now = date('Y-m-d H:i:s');
        
        $query = "SELECT id FROM refresh_tokens 
                  WHERE user_id = ? AND token = ? AND expires_at > ? AND revoked = 0";
        $result = $db->query($query, [$userId, $token, $now]);
        
        return !empty($result);
    }

    /**
     * Revoke refresh token
     */
    private function revokeRefreshToken($token)
    {
        $db = Database::getInstance();
        $query = "UPDATE refresh_tokens SET revoked = 1 WHERE token = ?";
        $db->execute($query, [$token]);
    }

    /**
     * Get customer role ID
     */
    private function getCustomerRoleId()
    {
        $db = Database::getInstance();
        $query = "SELECT id FROM roles WHERE name = 'customer' LIMIT 1";
        $result = $db->query($query);
        
        return $result[0]['id'] ?? null;
    }

    /**
     * Change password
     */
    public function changePassword($userId, $oldPassword, $newPassword)
    {
        try {
            $user = $this->userRepo->findById($userId);

            if (!$user) {
                throw new NotFoundException('User not found');
            }

            // Verify old password
            if (!password_verify($oldPassword, $user->getPassword())) {
                throw new ValidationException('Current password is incorrect');
            }

            // Hash new password
            $hashedPassword = password_hash($newPassword, PASSWORD_BCRYPT, ['cost' => 10]);

            // Update password
            $this->userRepo->update($userId, ['password' => $hashedPassword]);

            $this->logger->info('Password changed', ['user_id' => $userId]);

            return true;

        } catch (Exception $e) {
            $this->logger->error('Password change error', [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }
}
