<?php
/**
 * Payment Repository
 * Handles database operations for payments
 */
class PaymentRepository
{
    private $db;
    private $table = 'payments';

    public function __construct()
    {
        $this->db = Database::getInstance();
    }

    /**
     * Find payment by ID
     */
    public function findById($id)
    {
        $query = "SELECT * FROM {$this->table} WHERE id = ? AND deleted_at IS NULL";
        $result = $this->db->query($query, [$id]);
        
        if (empty($result)) {
            return null;
        }
        
        $payment = new Payment($result[0]);
        $this->loadRelations($payment);
        
        return $payment;
    }

    /**
     * Find payment by payment code
     */
    public function findByPaymentCode($code)
    {
        $query = "SELECT * FROM {$this->table} WHERE payment_code = ? AND deleted_at IS NULL";
        $result = $this->db->query($query, [$code]);
        
        if (empty($result)) {
            return null;
        }
        
        $payment = new Payment($result[0]);
        $this->loadRelations($payment);
        
        return $payment;
    }

    /**
     * Find payment by external ID
     */
    public function findByExternalId($externalId)
    {
        $query = "SELECT * FROM {$this->table} WHERE external_id = ? AND deleted_at IS NULL";
        $result = $this->db->query($query, [$externalId]);
        
        if (empty($result)) {
            return null;
        }
        
        return new Payment($result[0]);
    }

    /**
     * Get payments by invoice ID
     */
    public function getByInvoiceId($invoiceId)
    {
        $query = "SELECT * FROM {$this->table} WHERE invoice_id = ? AND deleted_at IS NULL
                  ORDER BY created_at DESC";
        $results = $this->db->query($query, [$invoiceId]);
        $payments = [];
        
        foreach ($results as $row) {
            $payments[] = new Payment($row);
        }
        
        return $payments;
    }

    /**
     * Create new payment
     */
    public function create($data)
    {
        $id = $this->db->generateUUID();
        $now = date('Y-m-d H:i:s');
        
        // Generate payment code
        $paymentCode = $this->generatePaymentCode();
        
        $query = "INSERT INTO {$this->table} 
                  (id, payment_code, invoice_id, gateway_id, channel_id, amount, admin_fee,
                   total_amount, status, payment_url, external_id, expired_at, created_at, updated_at)
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $id,
            $paymentCode,
            $data['invoice_id'],
            $data['gateway_id'],
            $data['channel_id'] ?? null,
            $data['amount'],
            $data['admin_fee'] ?? 0,
            $data['total_amount'],
            $data['status'] ?? 'pending',
            $data['payment_url'] ?? null,
            $data['external_id'] ?? null,
            $data['expired_at'] ?? null,
            $now,
            $now
        ];
        
        $this->db->execute($query, $params);
        
        return $this->findById($id);
    }

    /**
     * Update payment
     */
    public function update($id, $data)
    {
        $now = date('Y-m-d H:i:s');
        $fields = [];
        $params = [];
        
        $allowedFields = ['status', 'payment_url', 'external_id', 'paid_at', 'expired_at'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $params[] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return $this->findById($id);
        }
        
        $fields[] = "updated_at = ?";
        $params[] = $now;
        $params[] = $id;
        
        $query = "UPDATE {$this->table} SET " . implode(', ', $fields) . " WHERE id = ?";
        $this->db->execute($query, $params);
        
        return $this->findById($id);
    }

    /**
     * Mark payment as paid
     */
    public function markAsPaid($id)
    {
        $now = date('Y-m-d H:i:s');
        $query = "UPDATE {$this->table} SET status = 'paid', paid_at = ?, updated_at = ? WHERE id = ?";
        $this->db->execute($query, [$now, $now, $id]);
        
        return $this->findById($id);
    }

    /**
     * Mark payment as failed
     */
    public function markAsFailed($id, $reason = null)
    {
        $now = date('Y-m-d H:i:s');
        $query = "UPDATE {$this->table} SET status = 'failed', updated_at = ? WHERE id = ?";
        $this->db->execute($query, [$now, $id]);
        
        return $this->findById($id);
    }

    /**
     * Get all payments with pagination
     */
    public function getAll($page = 1, $perPage = 20, $filters = [])
    {
        $offset = ($page - 1) * $perPage;
        $where = ["deleted_at IS NULL"];
        $params = [];
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['gateway_id'])) {
            $where[] = "gateway_id = ?";
            $params[] = $filters['gateway_id'];
        }
        
        if (!empty($filters['invoice_id'])) {
            $where[] = "invoice_id = ?";
            $params[] = $filters['invoice_id'];
        }
        
        if (!empty($filters['from_date'])) {
            $where[] = "created_at >= ?";
            $params[] = $filters['from_date'];
        }
        
        if (!empty($filters['to_date'])) {
            $where[] = "created_at <= ?";
            $params[] = $filters['to_date'];
        }
        
        $whereClause = implode(' AND ', $where);
        
        $query = "SELECT * FROM {$this->table} WHERE {$whereClause} 
                  ORDER BY created_at DESC LIMIT ? OFFSET ?";
        $params[] = $perPage;
        $params[] = $offset;
        
        $results = $this->db->query($query, $params);
        $payments = [];
        
        foreach ($results as $row) {
            $payment = new Payment($row);
            $this->loadRelations($payment);
            $payments[] = $payment;
        }
        
        return $payments;
    }

    /**
     * Count payments
     */
    public function count($filters = [])
    {
        $where = ["deleted_at IS NULL"];
        $params = [];
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['gateway_id'])) {
            $where[] = "gateway_id = ?";
            $params[] = $filters['gateway_id'];
        }
        
        $whereClause = implode(' AND ', $where);
        $query = "SELECT COUNT(*) as total FROM {$this->table} WHERE {$whereClause}";
        $result = $this->db->query($query, $params);
        
        return $result[0]['total'] ?? 0;
    }

    /**
     * Get expired payments
     */
    public function getExpiredPayments()
    {
        $now = date('Y-m-d H:i:s');
        $query = "SELECT * FROM {$this->table} 
                  WHERE status = 'pending' AND expired_at < ? AND deleted_at IS NULL
                  ORDER BY expired_at ASC";
        
        $results = $this->db->query($query, [$now]);
        $payments = [];
        
        foreach ($results as $row) {
            $payments[] = new Payment($row);
        }
        
        return $payments;
    }

    /**
     * Load payment relations
     */
    private function loadRelations($payment)
    {
        // Load invoice
        $invoiceQuery = "SELECT * FROM invoices WHERE id = ? AND deleted_at IS NULL";
        $invoiceResult = $this->db->query($invoiceQuery, [$payment->getInvoiceId()]);
        
        if (!empty($invoiceResult)) {
            $payment->setInvoice($invoiceResult[0]);
        }

        // Load gateway
        $gatewayQuery = "SELECT * FROM payment_gateways WHERE id = ?";
        $gatewayResult = $this->db->query($gatewayQuery, [$payment->getGatewayId()]);
        
        if (!empty($gatewayResult)) {
            $payment->setGateway($gatewayResult[0]);
        }
    }

    /**
     * Generate unique payment code
     */
    private function generatePaymentCode()
    {
        $prefix = 'PAY';
        $year = date('Y');
        $month = date('m');
        $day = date('d');
        
        // Get the last payment code for today
        $query = "SELECT payment_code FROM {$this->table} 
                  WHERE payment_code LIKE ? 
                  ORDER BY payment_code DESC LIMIT 1";
        $pattern = "{$prefix}{$year}{$month}{$day}%";
        $result = $this->db->query($query, [$pattern]);
        
        if (empty($result)) {
            $sequence = 1;
        } else {
            $lastCode = $result[0]['payment_code'];
            $sequence = (int)substr($lastCode, -4) + 1;
        }
        
        return sprintf("%s%s%s%s%04d", $prefix, $year, $month, $day, $sequence);
    }

    /**
     * Record webhook log
     */
    public function recordWebhook($paymentId, $gatewayId, $payload, $signature = null)
    {
        $id = $this->db->generateUUID();
        $query = "INSERT INTO webhook_logs 
                  (id, payment_id, gateway_id, payload, signature, created_at)
                  VALUES (?, ?, ?, ?, ?, ?)";
        
        $this->db->execute($query, [
            $id,
            $paymentId,
            $gatewayId,
            json_encode($payload),
            $signature,
            date('Y-m-d H:i:s')
        ]);
    }
}
