<?php
/**
 * Invoice Repository
 * Handles database operations for invoices
 */
class InvoiceRepository
{
    private $db;
    private $table = 'invoices';

    public function __construct()
    {
        $this->db = Database::getInstance();
    }

    /**
     * Find invoice by ID
     */
    public function findById($id)
    {
        $query = "SELECT * FROM {$this->table} WHERE id = ? AND deleted_at IS NULL";
        $result = $this->db->query($query, [$id]);
        
        if (empty($result)) {
            return null;
        }
        
        $invoice = new Invoice($result[0]);
        $this->loadRelations($invoice);
        
        return $invoice;
    }

    /**
     * Find invoice by invoice number
     */
    public function findByInvoiceNumber($invoiceNumber)
    {
        $query = "SELECT * FROM {$this->table} WHERE invoice_number = ? AND deleted_at IS NULL";
        $result = $this->db->query($query, [$invoiceNumber]);
        
        if (empty($result)) {
            return null;
        }
        
        $invoice = new Invoice($result[0]);
        $this->loadRelations($invoice);
        
        return $invoice;
    }

    /**
     * Get invoices by customer ID
     */
    public function getByCustomerId($customerId, $page = 1, $perPage = 20, $filters = [])
    {
        $offset = ($page - 1) * $perPage;
        $where = ["customer_id = ?", "deleted_at IS NULL"];
        $params = [$customerId];
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
        }
        
        $whereClause = implode(' AND ', $where);
        
        $query = "SELECT * FROM {$this->table} WHERE {$whereClause} 
                  ORDER BY issue_date DESC LIMIT ? OFFSET ?";
        $params[] = $perPage;
        $params[] = $offset;
        
        $results = $this->db->query($query, $params);
        $invoices = [];
        
        foreach ($results as $row) {
            $invoice = new Invoice($row);
            $this->loadRelations($invoice);
            $invoices[] = $invoice;
        }
        
        return $invoices;
    }

    /**
     * Create new invoice
     */
    public function create($data)
    {
        $id = $this->db->generateUUID();
        $now = date('Y-m-d H:i:s');
        
        // Generate invoice number
        $invoiceNumber = $this->generateInvoiceNumber();
        
        $query = "INSERT INTO {$this->table} 
                  (id, invoice_number, customer_id, billing_cycle_id, issue_date, due_date,
                   subtotal, discount_amount, tax_amount, total_amount, status, created_at, updated_at)
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $id,
            $invoiceNumber,
            $data['customer_id'],
            $data['billing_cycle_id'] ?? null,
            $data['issue_date'] ?? date('Y-m-d'),
            $data['due_date'],
            $data['subtotal'],
            $data['discount_amount'] ?? 0,
            $data['tax_amount'] ?? 0,
            $data['total_amount'],
            $data['status'] ?? 'pending',
            $now,
            $now
        ];
        
        $this->db->execute($query, $params);
        
        // Create invoice items if provided
        if (!empty($data['items'])) {
            foreach ($data['items'] as $item) {
                $this->createInvoiceItem($id, $item);
            }
        }
        
        return $this->findById($id);
    }

    /**
     * Update invoice
     */
    public function update($id, $data)
    {
        $now = date('Y-m-d H:i:s');
        $fields = [];
        $params = [];
        
        $allowedFields = ['due_date', 'subtotal', 'discount_amount', 'tax_amount', 
                          'total_amount', 'status', 'paid_at'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $params[] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return $this->findById($id);
        }
        
        $fields[] = "updated_at = ?";
        $params[] = $now;
        $params[] = $id;
        
        $query = "UPDATE {$this->table} SET " . implode(', ', $fields) . " WHERE id = ?";
        $this->db->execute($query, $params);
        
        return $this->findById($id);
    }

    /**
     * Mark invoice as paid
     */
    public function markAsPaid($id)
    {
        $now = date('Y-m-d H:i:s');
        $query = "UPDATE {$this->table} SET status = 'paid', paid_at = ?, updated_at = ? WHERE id = ?";
        $this->db->execute($query, [$now, $now, $id]);
        
        return $this->findById($id);
    }

    /**
     * Mark invoice as overdue
     */
    public function markAsOverdue($id)
    {
        $now = date('Y-m-d H:i:s');
        $query = "UPDATE {$this->table} SET status = 'overdue', updated_at = ? WHERE id = ?";
        $this->db->execute($query, [$now, $id]);
        
        return $this->findById($id);
    }

    /**
     * Get all invoices with pagination
     */
    public function getAll($page = 1, $perPage = 20, $filters = [])
    {
        $offset = ($page - 1) * $perPage;
        $where = ["deleted_at IS NULL"];
        $params = [];
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['customer_id'])) {
            $where[] = "customer_id = ?";
            $params[] = $filters['customer_id'];
        }
        
        if (!empty($filters['from_date'])) {
            $where[] = "issue_date >= ?";
            $params[] = $filters['from_date'];
        }
        
        if (!empty($filters['to_date'])) {
            $where[] = "issue_date <= ?";
            $params[] = $filters['to_date'];
        }
        
        $whereClause = implode(' AND ', $where);
        
        $query = "SELECT * FROM {$this->table} WHERE {$whereClause} 
                  ORDER BY issue_date DESC LIMIT ? OFFSET ?";
        $params[] = $perPage;
        $params[] = $offset;
        
        $results = $this->db->query($query, $params);
        $invoices = [];
        
        foreach ($results as $row) {
            $invoice = new Invoice($row);
            $this->loadRelations($invoice);
            $invoices[] = $invoice;
        }
        
        return $invoices;
    }

    /**
     * Count invoices
     */
    public function count($filters = [])
    {
        $where = ["deleted_at IS NULL"];
        $params = [];
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['customer_id'])) {
            $where[] = "customer_id = ?";
            $params[] = $filters['customer_id'];
        }
        
        $whereClause = implode(' AND ', $where);
        $query = "SELECT COUNT(*) as total FROM {$this->table} WHERE {$whereClause}";
        $result = $this->db->query($query, $params);
        
        return $result[0]['total'] ?? 0;
    }

    /**
     * Get overdue invoices
     */
    public function getOverdueInvoices()
    {
        $today = date('Y-m-d');
        $query = "SELECT * FROM {$this->table} 
                  WHERE status = 'pending' AND due_date < ? AND deleted_at IS NULL
                  ORDER BY due_date ASC";
        
        $results = $this->db->query($query, [$today]);
        $invoices = [];
        
        foreach ($results as $row) {
            $invoices[] = new Invoice($row);
        }
        
        return $invoices;
    }

    /**
     * Create invoice item
     */
    private function createInvoiceItem($invoiceId, $itemData)
    {
        $id = $this->db->generateUUID();
        $now = date('Y-m-d H:i:s');
        
        $query = "INSERT INTO invoice_items 
                  (id, invoice_id, description, quantity, unit_price, discount, tax, subtotal, created_at)
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $id,
            $invoiceId,
            $itemData['description'],
            $itemData['quantity'] ?? 1,
            $itemData['unit_price'],
            $itemData['discount'] ?? 0,
            $itemData['tax'] ?? 0,
            $itemData['subtotal'],
            $now
        ];
        
        $this->db->execute($query, $params);
    }

    /**
     * Load invoice relations (customer, items, payments)
     */
    private function loadRelations($invoice)
    {
        // Load customer
        $customerQuery = "SELECT * FROM customers WHERE id = ? AND deleted_at IS NULL";
        $customerResult = $this->db->query($customerQuery, [$invoice->getCustomerId()]);
        
        if (!empty($customerResult)) {
            $invoice->setCustomer($customerResult[0]);
        }

        // Load invoice items
        $itemsQuery = "SELECT * FROM invoice_items WHERE invoice_id = ? ORDER BY created_at";
        $items = $this->db->query($itemsQuery, [$invoice->getId()]);
        $invoice->setItems($items);

        // Load payments
        $paymentsQuery = "SELECT * FROM payments WHERE invoice_id = ? ORDER BY created_at DESC";
        $payments = $this->db->query($paymentsQuery, [$invoice->getId()]);
        $invoice->setPayments($payments);
    }

    /**
     * Generate unique invoice number
     */
    private function generateInvoiceNumber()
    {
        $prefix = 'INV';
        $year = date('Y');
        $month = date('m');
        
        // Get the last invoice number for this month
        $query = "SELECT invoice_number FROM {$this->table} 
                  WHERE invoice_number LIKE ? 
                  ORDER BY invoice_number DESC LIMIT 1";
        $pattern = "{$prefix}-{$year}{$month}-%";
        $result = $this->db->query($query, [$pattern]);
        
        if (empty($result)) {
            $sequence = 1;
        } else {
            $lastNumber = $result[0]['invoice_number'];
            $sequence = (int)substr($lastNumber, -5) + 1;
        }
        
        return sprintf("%s-%s%s-%05d", $prefix, $year, $month, $sequence);
    }
}
