<?php
/**
 * Rate Limit Middleware
 * Prevent API abuse with rate limiting
 */

class RateLimitMiddleware {
    private static $storage = [];
    
    public static function handle(Request $request): void {
        $ip = $request->getIp();
        $key = "rate_limit:{$ip}";
        $now = time();
        $window = RATE_LIMIT_WINDOW;
        $limit = RATE_LIMIT_REQUESTS;
        
        // Get or initialize rate limit data
        if (!isset(self::$storage[$key])) {
            self::$storage[$key] = [
                'count' => 0,
                'reset_at' => $now + $window
            ];
        }
        
        $data = self::$storage[$key];
        
        // Reset if window expired
        if ($now >= $data['reset_at']) {
            self::$storage[$key] = [
                'count' => 1,
                'reset_at' => $now + $window
            ];
            return;
        }
        
        // Increment counter
        $data['count']++;
        self::$storage[$key] = $data;
        
        // Check limit
        if ($data['count'] > $limit) {
            $retryAfter = $data['reset_at'] - $now;
            
            header("X-RateLimit-Limit: {$limit}");
            header("X-RateLimit-Remaining: 0");
            header("X-RateLimit-Reset: {$data['reset_at']}");
            header("Retry-After: {$retryAfter}");
            
            Logger::warning('Rate limit exceeded', [
                'ip' => $ip,
                'uri' => $request->getUri()
            ]);
            
            Response::error('Too many requests. Please try again later.', null, 429);
        }
        
        // Add rate limit headers
        $remaining = max(0, $limit - $data['count']);
        header("X-RateLimit-Limit: {$limit}");
        header("X-RateLimit-Remaining: {$remaining}");
        header("X-RateLimit-Reset: {$data['reset_at']}");
    }
}
