<?php
/**
 * Auth Middleware
 * Verify JWT token and authenticate user
 */

class AuthMiddleware {
    public static function handle(Request $request): void {
        $payload = JWT::verifyRequest($request);
        
        if (!$payload) {
            Response::unauthorized('Invalid or expired token');
        }
        
        // Get user from database
        $db = Database::getInstance();
        $user = $db->fetchOne(
            "SELECT u.*, r.name as role_name 
             FROM users u 
             JOIN roles r ON u.role_id = r.id 
             WHERE u.id = :id AND u.is_active = 1",
            ['id' => $payload['user_id']]
        );
        
        if (!$user) {
            Response::unauthorized('User not found or inactive');
        }
        
        // Store user in request (you can use a global or static property)
        self::$currentUser = $user;
        
        // Log access
        Logger::debug('Authenticated user access', [
            'user_id' => $user['id'],
            'role' => $user['role_name'],
            'uri' => $request->getUri()
        ]);
    }
    
    private static $currentUser = null;
    
    /**
     * Get current authenticated user
     */
    public static function user(): ?array {
        return self::$currentUser;
    }
    
    /**
     * Check if user has role
     */
    public static function hasRole(string $role): bool {
        $user = self::user();
        return $user && $user['role_name'] === $role;
    }
    
    /**
     * Check if user has permission
     */
    public static function hasPermission(string $permission): bool {
        $user = self::user();
        
        if (!$user) {
            return false;
        }
        
        $db = Database::getInstance();
        
        // Check role permissions
        $count = $db->fetchOne(
            "SELECT COUNT(*) as count FROM role_permissions rp
             JOIN permissions p ON rp.permission_id = p.id
             WHERE rp.role_id = :role_id AND p.name = :permission",
            ['role_id' => $user['role_id'], 'permission' => $permission]
        );
        
        if ($count['count'] > 0) {
            return true;
        }
        
        // Check user-specific permissions
        $count = $db->fetchOne(
            "SELECT COUNT(*) as count FROM user_permissions up
             JOIN permissions p ON up.permission_id = p.id
             WHERE up.user_id = :user_id AND p.name = :permission",
            ['user_id' => $user['id'], 'permission' => $permission]
        );
        
        return $count['count'] > 0;
    }
}
