<?php
/**
 * Router Class
 * RESTful routing with middleware support
 */

class Router {
    private $routes = [];
    private $middleware = [];
    private $groupPrefix = '';
    private $groupMiddleware = [];
    
    /**
     * Add GET route
     */
    public function get(string $path, $handler, array $middleware = []): void {
        $this->addRoute('GET', $path, $handler, $middleware);
    }
    
    /**
     * Add POST route
     */
    public function post(string $path, $handler, array $middleware = []): void {
        $this->addRoute('POST', $path, $handler, $middleware);
    }
    
    /**
     * Add PUT route
     */
    public function put(string $path, $handler, array $middleware = []): void {
        $this->addRoute('PUT', $path, $handler, $middleware);
    }
    
    /**
     * Add DELETE route
     */
    public function delete(string $path, $handler, array $middleware = []): void {
        $this->addRoute('DELETE', $path, $handler, $middleware);
    }
    
    /**
     * Add PATCH route
     */
    public function patch(string $path, $handler, array $middleware = []): void {
        $this->addRoute('PATCH', $path, $handler, $middleware);
    }
    
    /**
     * Add route
     */
    private function addRoute(string $method, string $path, $handler, array $middleware = []): void {
        $fullPath = $this->groupPrefix . $path;
        $allMiddleware = array_merge($this->groupMiddleware, $middleware);
        
        $this->routes[] = [
            'method' => $method,
            'path' => $fullPath,
            'handler' => $handler,
            'middleware' => $allMiddleware,
            'pattern' => $this->convertToPattern($fullPath)
        ];
    }
    
    /**
     * Convert path to regex pattern
     */
    private function convertToPattern(string $path): string {
        $pattern = preg_replace('/\{([a-zA-Z0-9_]+)\}/', '(?P<$1>[^/]+)', $path);
        return '#^' . $pattern . '$#';
    }
    
    /**
     * Group routes with prefix and middleware
     */
    public function group(array $attributes, callable $callback): void {
        $previousPrefix = $this->groupPrefix;
        $previousMiddleware = $this->groupMiddleware;
        
        if (isset($attributes['prefix'])) {
            $this->groupPrefix = $previousPrefix . $attributes['prefix'];
        }
        
        if (isset($attributes['middleware'])) {
            $this->groupMiddleware = array_merge(
                $previousMiddleware,
                is_array($attributes['middleware']) ? $attributes['middleware'] : [$attributes['middleware']]
            );
        }
        
        $callback($this);
        
        $this->groupPrefix = $previousPrefix;
        $this->groupMiddleware = $previousMiddleware;
    }
    
    /**
     * Register global middleware
     */
    public function middleware(string $name, $handler): void {
        $this->middleware[$name] = $handler;
    }
    
    /**
     * Dispatch request
     */
    public function dispatch(Request $request): void {
        $method = $request->getMethod();
        $uri = $request->getUri();
        
        foreach ($this->routes as $route) {
            // Handle OPTIONS request for CORS (only for valid routes)
            if ($method === 'OPTIONS' && preg_match($route['pattern'], $uri)) {
                Response::success(null, 'OK', 200);
            }
            
            if ($route['method'] !== $method) {
                continue;
            }
            
            if (preg_match($route['pattern'], $uri, $matches)) {
                // Extract route parameters
                $params = array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);
                $request->setRouteParams($params);
                
                try {
                    // Execute middleware
                    foreach ($route['middleware'] as $middlewareName) {
                        if (isset($this->middleware[$middlewareName])) {
                            $middlewareHandler = $this->middleware[$middlewareName];
                            $middlewareHandler($request);
                        }
                    }
                    
                    // Execute handler
                    $this->executeHandler($route['handler'], $request);
                    return;
                    
                } catch (Exception $e) {
                    Logger::error('Route execution failed', [
                        'uri' => $uri,
                        'method' => $method,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    
                    if (APP_DEBUG) {
                        Response::serverError($e->getMessage());
                    } else {
                        Response::serverError('Internal server error');
                    }
                }
            }
        }
        
        // No route found
        Response::notFound('Route not found');
    }
    
    /**
     * Execute route handler
     */
    private function executeHandler($handler, Request $request): void {
        if (is_string($handler)) {
            // Controller@method format
            if (strpos($handler, '@') !== false) {
                list($controller, $method) = explode('@', $handler);
                
                $controllerClass = $controller;
                if (!class_exists($controllerClass)) {
                    throw new Exception("Controller {$controllerClass} not found");
                }
                
                $controllerInstance = new $controllerClass();
                
                if (!method_exists($controllerInstance, $method)) {
                    throw new Exception("Method {$method} not found in {$controllerClass}");
                }
                
                $controllerInstance->$method($request);
            }
        } elseif (is_callable($handler)) {
            // Closure
            $handler($request);
        }
    }
}
