<?php
/**
 * Notification Controller
 * Handles push notification endpoints
 */
class NotificationController
{
    private $request;
    private $validator;
    private $logger;
    private $notificationService;
    private $db;

    public function __construct()
    {
        $this->request = new Request();
        $this->validator = new Validator();
        $this->logger = Logger::getInstance();
        $this->notificationService = new NotificationService();
        $this->db = Database::getInstance();
    }

    /**
     * Register FCM token
     * POST /api/notifications/register-token
     */
    public function registerToken()
    {
        try {
            $user = $this->request->getUser();
            $data = $this->request->getBody();

            // Validate input
            $rules = [
                'fcm_token' => 'required|string',
                'device_type' => 'string',
                'device_model' => 'string'
            ];

            $errors = $this->validator->validate($data, $rules);
            
            if (!empty($errors)) {
                return Response::validationError($errors);
            }

            $deviceInfo = [
                'type' => $data['device_type'] ?? 'mobile',
                'model' => $data['device_model'] ?? 'unknown'
            ];

            $result = $this->notificationService->registerToken(
                $user['id'],
                $data['fcm_token'],
                $deviceInfo
            );

            if ($result) {
                return Response::success([], 'Token registered successfully');
            } else {
                return Response::serverError('Failed to register token');
            }

        } catch (Exception $e) {
            $this->logger->error('Register token error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to register token');
        }
    }

    /**
     * Unregister FCM token
     * POST /api/notifications/unregister-token
     */
    public function unregisterToken()
    {
        try {
            $data = $this->request->getBody();

            // Validate input
            $rules = [
                'fcm_token' => 'required|string'
            ];

            $errors = $this->validator->validate($data, $rules);
            
            if (!empty($errors)) {
                return Response::validationError($errors);
            }

            $result = $this->notificationService->unregisterToken($data['fcm_token']);

            if ($result) {
                return Response::success([], 'Token unregistered successfully');
            } else {
                return Response::serverError('Failed to unregister token');
            }

        } catch (Exception $e) {
            $this->logger->error('Unregister token error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to unregister token');
        }
    }

    /**
     * Get user notifications
     * GET /api/notifications
     */
    public function index()
    {
        try {
            $user = $this->request->getUser();
            $page = $this->request->getQueryParam('page', 1);
            $perPage = $this->request->getQueryParam('per_page', 20);
            $unreadOnly = $this->request->getQueryParam('unread_only', false);
            
            $offset = ($page - 1) * $perPage;
            
            // Get customer ID
            $customerQuery = "SELECT id FROM customers WHERE user_id = ?";
            $customerResult = $this->db->query($customerQuery, [$user['id']]);
            
            if (empty($customerResult)) {
                return Response::success([
                    'notifications' => [],
                    'pagination' => [
                        'current_page' => 1,
                        'per_page' => $perPage,
                        'total' => 0,
                        'total_pages' => 0
                    ]
                ]);
            }
            
            $customerId = $customerResult[0]['id'];
            
            // Build query
            $query = "SELECT * FROM notifications WHERE customer_id = ?";
            $params = [$customerId];
            
            if ($unreadOnly) {
                $query .= " AND is_read = 0";
            }
            
            $query .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
            $params[] = $perPage;
            $params[] = $offset;
            
            $notifications = $this->db->query($query, $params);
            
            // Get total count
            $countQuery = "SELECT COUNT(*) as total FROM notifications WHERE customer_id = ?";
            $countParams = [$customerId];
            
            if ($unreadOnly) {
                $countQuery .= " AND is_read = 0";
            }
            
            $totalResult = $this->db->query($countQuery, $countParams);
            $total = $totalResult[0]['total'] ?? 0;
            
            return Response::success([
                'notifications' => $notifications,
                'pagination' => [
                    'current_page' => (int)$page,
                    'per_page' => (int)$perPage,
                    'total' => (int)$total,
                    'total_pages' => ceil($total / $perPage)
                ]
            ]);

        } catch (Exception $e) {
            $this->logger->error('Get notifications error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve notifications');
        }
    }

    /**
     * Mark notification as read
     * PUT /api/notifications/{id}/read
     */
    public function markAsRead($id)
    {
        try {
            $user = $this->request->getUser();
            
            // Get customer ID
            $customerQuery = "SELECT id FROM customers WHERE user_id = ?";
            $customerResult = $this->db->query($customerQuery, [$user['id']]);
            
            if (empty($customerResult)) {
                return Response::notFound('Customer not found');
            }
            
            $customerId = $customerResult[0]['id'];
            
            // Update notification
            $query = "UPDATE notifications 
                     SET is_read = 1, read_at = NOW() 
                     WHERE id = ? AND customer_id = ?";
            
            $this->db->execute($query, [$id, $customerId]);
            
            return Response::success([], 'Notification marked as read');

        } catch (Exception $e) {
            $this->logger->error('Mark notification as read error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to mark notification as read');
        }
    }
}
