<?php
/**
 * Invoice Controller
 * Handles invoice endpoints
 */
class InvoiceController
{
    private $billingService;
    private $validator;
    private $request;
    private $logger;

    public function __construct()
    {
        $this->billingService = new BillingService();
        $this->validator = new Validator();
        $this->request = new Request();
        $this->logger = Logger::getInstance();
    }

    /**
     * Get customer invoices
     * GET /api/invoices/my
     */
    public function my()
    {
        try {
            $userId = $this->request->getAttribute('user_id');

            if (!$userId) {
                return Response::unauthorized('Not authenticated');
            }

            // Get customer
            $customerService = new CustomerService();
            $customer = $customerService->getByUserId($userId);

            $page = $this->request->getQueryParam('page', 1);
            $perPage = $this->request->getQueryParam('per_page', 20);
            $filters = [
                'status' => $this->request->getQueryParam('status')
            ];

            $result = $this->billingService->getCustomerInvoices(
                $customer->getId(),
                $page,
                $perPage,
                $filters
            );

            return Response::paginated($result['data'], $result['pagination']);

        } catch (NotFoundException $e) {
            return Response::notFound($e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Get my invoices error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve invoices');
        }
    }

    /**
     * Get invoice by ID
     * GET /api/invoices/{id}
     */
    public function show($id)
    {
        try {
            $invoice = $this->billingService->getInvoice($id);

            // Check authorization
            $userId = $this->request->getAttribute('user_id');
            $userRole = $this->request->getAttribute('role_id');
            
            // Allow if admin or owner
            if ($userRole !== 'admin') {
                $customerService = new CustomerService();
                $customer = $customerService->getByUserId($userId);
                
                if ($invoice->getCustomerId() !== $customer->getId()) {
                    return Response::forbidden('Not authorized to view this invoice');
                }
            }

            return Response::success($invoice->toArray(), 'Invoice retrieved successfully');

        } catch (NotFoundException $e) {
            return Response::notFound($e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Get invoice error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve invoice');
        }
    }

    /**
     * Get all invoices (Admin only)
     * GET /api/invoices
     */
    public function index()
    {
        try {
            $page = $this->request->getQueryParam('page', 1);
            $perPage = $this->request->getQueryParam('per_page', 20);
            $filters = [
                'status' => $this->request->getQueryParam('status'),
                'customer_id' => $this->request->getQueryParam('customer_id'),
                'from_date' => $this->request->getQueryParam('from_date'),
                'to_date' => $this->request->getQueryParam('to_date')
            ];

            $result = $this->billingService->getAllInvoices($page, $perPage, $filters);

            return Response::paginated($result['data'], $result['pagination']);

        } catch (Exception $e) {
            $this->logger->error('Get invoices error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve invoices');
        }
    }

    /**
     * Create invoice (Admin only)
     * POST /api/invoices
     */
    public function create()
    {
        try {
            $data = $this->request->getBody();

            // Validate input
            $rules = [
                'customer_id' => 'required|uuid',
                'due_date' => 'required|date',
                'subtotal' => 'required|numeric',
                'items' => 'required|array'
            ];

            $errors = $this->validator->validate($data, $rules);
            
            if (!empty($errors)) {
                return Response::validationError($errors);
            }

            // Generate invoice
            $invoice = $this->billingService->generateInvoice($data);

            return Response::created($invoice->toArray(), 'Invoice created successfully');

        } catch (NotFoundException $e) {
            return Response::notFound($e->getMessage());
        } catch (ValidationException $e) {
            return Response::validationError($e->getDetails(), $e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Create invoice error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to create invoice');
        }
    }

    /**
     * Get dashboard statistics (Admin only)
     * GET /api/invoices/stats
     */
    public function stats()
    {
        try {
            $stats = $this->billingService->getDashboardStats();

            return Response::success($stats, 'Statistics retrieved successfully');

        } catch (Exception $e) {
            $this->logger->error('Get invoice stats error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve statistics');
        }
    }

    /**
     * Download invoice as PDF
     * GET /api/invoices/{id}/download
     */
    public function download($id)
    {
        try {
            $invoice = $this->billingService->getInvoice($id);

            // Check authorization
            $userId = $this->request->getAttribute('user_id');
            $userRole = $this->request->getAttribute('role_id');
            
            // Allow if admin or owner
            if ($userRole !== 'admin') {
                $customerService = new CustomerService();
                $customer = $customerService->getByUserId($userId);
                
                if ($invoice->getCustomerId() !== $customer->getId()) {
                    return Response::forbidden('Not authorized to download this invoice');
                }
            }

            // Generate PDF
            $pdfService = new InvoicePdfService();
            $pdfContent = $pdfService->generateInvoicePdf($invoice);

            // Set headers for PDF download
            header('Content-Type: application/pdf');
            header('Content-Disposition: attachment; filename="invoice-' . $invoice->getInvoiceNumber() . '.pdf"');
            header('Content-Length: ' . strlen($pdfContent));
            header('Cache-Control: private, max-age=0, must-revalidate');
            header('Pragma: public');
            
            echo $pdfContent;
            exit;

        } catch (NotFoundException $e) {
            return Response::notFound($e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Download invoice error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to download invoice');
        }
    }
}
