<?php
/**
 * Customer Controller
 * Handles customer endpoints
 */
class CustomerController
{
    private $customerService;
    private $validator;
    private $request;
    private $logger;

    public function __construct()
    {
        $this->customerService = new CustomerService();
        $this->validator = new Validator();
        $this->request = new Request();
        $this->logger = Logger::getInstance();
    }

    /**
     * Register customer
     * POST /api/customers/register
     */
    public function register()
    {
        try {
            $data = $this->request->getBody();

            // Validate input
            $rules = [
                'email' => 'required|email',
                'password' => 'required|min:8',
                'full_name' => 'required|min:3',
                'phone' => 'phone',
                'address' => 'required',
                'city' => 'required',
                'latitude' => 'latitude',
                'longitude' => 'longitude'
            ];

            $errors = $this->validator->validate($data, $rules);
            
            if (!empty($errors)) {
                return Response::validationError($errors);
            }

            // Register customer
            $result = $this->customerService->register($data);

            return Response::created($result, 'Customer registered successfully');

        } catch (ValidationException $e) {
            return Response::validationError($e->getDetails(), $e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Customer registration error', ['error' => $e->getMessage()]);
            return Response::serverError('Registration failed');
        }
    }

    /**
     * Get current customer profile
     * GET /api/customers/me
     */
    public function me()
    {
        try {
            $userId = $this->request->getAttribute('user_id');

            if (!$userId) {
                return Response::unauthorized('Not authenticated');
            }

            $customer = $this->customerService->getByUserId($userId);

            return Response::success($customer->toArray(), 'Customer retrieved successfully');

        } catch (NotFoundException $e) {
            return Response::notFound($e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Get customer error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve customer');
        }
    }

    /**
     * Update customer profile
     * PUT /api/customers/{id}
     */
    public function update($id)
    {
        try {
            $data = $this->request->getBody();
            $userId = $this->request->getAttribute('user_id');

            // Verify ownership
            $customer = $this->customerService->getByUserId($userId);
            
            if ($customer->getId() !== $id) {
                return Response::forbidden('Not authorized to update this customer');
            }

            // Validate input
            $rules = [
                'full_name' => 'min:3',
                'phone' => 'phone',
                'latitude' => 'latitude',
                'longitude' => 'longitude'
            ];

            $errors = $this->validator->validate($data, $rules);
            
            if (!empty($errors)) {
                return Response::validationError($errors);
            }

            // Update profile
            $updated = $this->customerService->updateProfile($id, $data);

            return Response::success($updated->toArray(), 'Profile updated successfully');

        } catch (NotFoundException $e) {
            return Response::notFound($e->getMessage());
        } catch (AuthorizationException $e) {
            return Response::forbidden($e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Update customer error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to update profile');
        }
    }

    /**
     * Get all customers (Admin only)
     * GET /api/customers
     */
    public function index()
    {
        try {
            $page = $this->request->getQueryParam('page', 1);
            $perPage = $this->request->getQueryParam('per_page', 20);
            $filters = [
                'status' => $this->request->getQueryParam('status'),
                'city' => $this->request->getQueryParam('city'),
                'search' => $this->request->getQueryParam('search')
            ];

            $result = $this->customerService->getAll($page, $perPage, $filters);

            return Response::paginated($result['data'], $result['pagination']);

        } catch (Exception $e) {
            $this->logger->error('Get customers error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve customers');
        }
    }

    /**
     * Get customers by location (For collectors)
     * GET /api/customers/by-location
     */
    public function byLocation()
    {
        try {
            $latitude = $this->request->getQueryParam('latitude');
            $longitude = $this->request->getQueryParam('longitude');
            $radius = $this->request->getQueryParam('radius', 10);

            if (!$latitude || !$longitude) {
                return Response::badRequest('Latitude and longitude are required');
            }

            $customers = $this->customerService->getByLocation($latitude, $longitude, $radius);

            return Response::success($customers, 'Customers retrieved successfully');

        } catch (Exception $e) {
            $this->logger->error('Get customers by location error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve customers');
        }
    }
}
